﻿# =============================================================================
# 1. 관리자 권한 및 UI 라이브러리 로드
# =============================================================================
$currentPrincipal = [Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()
if (-not $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Start-Process powershell.exe -ArgumentList "-WindowStyle Hidden -File `"$($MyInvocation.MyCommand.Path)`"" -Verb RunAs
    exit
}

Add-Type -AssemblyName PresentationFramework
Add-Type -AssemblyName System.Windows.Forms

# =============================================================================
# 2. WPF XAML (디자인 유지 + 토글 스위치 추가)
# =============================================================================
[xml]$xaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        xmlns:x="http://schemas.microsoft.com/winfx/2006/xaml"
        Title="Win Refresh" Height="360" Width="260" WindowStyle="None" ResizeMode="NoResize"
        AllowsTransparency="True" Background="Transparent" Topmost="True" WindowStartupLocation="CenterScreen">
    <Window.Resources>
        <ControlTemplate x:Key="ToggleSwitchTemplate" TargetType="CheckBox">
            <Grid Background="Transparent">
                <Border Name="Track" Width="40" Height="22" CornerRadius="11" Background="#444444" BorderThickness="0">
                    <Ellipse Name="Knob" Width="18" Height="18" Fill="White" HorizontalAlignment="Left" Margin="2,0,0,0">
                        <Ellipse.Effect>
                            <DropShadowEffect ShadowDepth="1" BlurRadius="2" Opacity="0.3"/>
                        </Ellipse.Effect>
                    </Ellipse>
                </Border>
            </Grid>
            <ControlTemplate.Triggers>
                <Trigger Property="IsChecked" Value="True">
                    <Setter TargetName="Track" Property="Background" Value="#FF9500"/>
                    <Setter TargetName="Knob" Property="HorizontalAlignment" Value="Right"/>
                    <Setter TargetName="Knob" Property="Margin" Value="0,0,2,0"/>
                </Trigger>
            </ControlTemplate.Triggers>
        </ControlTemplate>
    </Window.Resources>

    <Border CornerRadius="12" Background="#1E1E1E" BorderBrush="#333333" BorderThickness="1">
        <Grid>
            <Grid.RowDefinitions><RowDefinition Height="30"/><RowDefinition Height="*"/><RowDefinition Height="55"/></Grid.RowDefinitions>
            
            <Border Grid.Row="0" Name="TitleBar" Background="#252525" CornerRadius="12,12,0,0" BorderBrush="#333" BorderThickness="0,0,0,1">
                <Grid>
                    <TextBlock Text="Win Refresh" Foreground="#777" VerticalAlignment="Center" HorizontalAlignment="Center" FontSize="11" FontWeight="SemiBold"/>
                    <Button Name="BtnClose" Content="×" HorizontalAlignment="Right" VerticalAlignment="Center" Margin="0,0,10,0" Foreground="#777" Background="Transparent" BorderThickness="0" FontWeight="Bold" Width="25" Height="25" FontSize="16" Cursor="Hand">
                        <Button.Template>
                            <ControlTemplate TargetType="Button">
                                <Border Background="Transparent">
                                    <ContentPresenter HorizontalAlignment="Center" VerticalAlignment="Center"/>
                                </Border>
                            </ControlTemplate>
                        </Button.Template>
                    </Button>
                </Grid>
            </Border>

            <StackPanel Grid.Row="1" VerticalAlignment="Top" HorizontalAlignment="Center" Margin="0,65,0,0">
                <TextBlock Name="StatusIcon" Text="🛡️" FontSize="40" HorizontalAlignment="Center" Margin="0,0,0,10" Foreground="#0078D7"/>
		<TextBlock Name="txt_OSInfo" Text="Checking..." FontSize="12" FontWeight="Light" Foreground="#AAAAAA" HorizontalAlignment="Center" Margin="0,15,0,0" />
                <TextBlock Name="ModeText" Text="일반 정리 모드" 
		   FontSize="13" FontWeight="Bold" Foreground="#999999" 
		   HorizontalAlignment="Center" Margin="0,15,0,20" 
		   Cursor="Help" VerticalAlignment="Center">
                    <TextBlock.ToolTip>
                        <ToolTip Background="#1E1E1E" Foreground="#DDDDDD" BorderBrush="#444" BorderThickness="1" HasDropShadow="True">
                            <TextBlock Name="TooltipContent" Text="목록 로딩 중..." FontFamily="Malgun Gothic" FontSize="11"/>
                        </ToolTip>
                    </TextBlock.ToolTip>
                </TextBlock>
                
                <StackPanel Orientation="Horizontal" HorizontalAlignment="Center" Margin="0,0,0,15">
                     <CheckBox Name="ToggleDeepClean" Template="{StaticResource ToggleSwitchTemplate}" Cursor="Hand"/>
                     <TextBlock Name="ToggleText" Text="정밀 정리 (오래 걸림)" Foreground="#666" VerticalAlignment="Center" Margin="10,0,0,0" FontSize="11"/>
                </StackPanel>

                <TextBlock Name="BigNumber" Text="0.00 MB" FontSize="36" FontWeight="Light" Foreground="White" HorizontalAlignment="Center" Visibility="Collapsed"/>
                <TextBlock Name="HintText" 
                           Text="※ WinSxS 등 시스템 정밀 정리는&#x0a;실행 시 추가로 확보됩니다." 
                           FontSize="10" Foreground="#4CC2FF" HorizontalAlignment="Center" 
                           TextAlignment="Center" Visibility="Collapsed" Margin="0,5,0,0"/>
                <TextBlock Name="LogText" Text="" FontSize="11" Foreground="#AAA" HorizontalAlignment="Center" Margin="0,10,0,0" TextTrimming="CharacterEllipsis" MaxWidth="240" TextAlignment="Center" Height="20"/>
            </StackPanel>

            <Grid Grid.Row="2" Background="Transparent" Margin="0,0,0,0">
                <Grid.RowDefinitions><RowDefinition Height="6"/><RowDefinition Height="*"/></Grid.RowDefinitions>
                <Grid.ColumnDefinitions><ColumnDefinition Width="*"/><ColumnDefinition Width="*"/></Grid.ColumnDefinitions>

                <ProgressBar Name="ProgressBar" Grid.Row="0" Grid.ColumnSpan="2" 
                             Background="#333" BorderThickness="0" Foreground="#4CC2FF" 
                             Value="0" Minimum="0" Maximum="100"/>

                <Button Name="SubBtn" Grid.Row="1" Grid.Column="0" Content="재검색" Visibility="Collapsed"
                        Background="#444444" Foreground="#DDDDDD" FontSize="14" FontWeight="Bold" Cursor="Hand" BorderThickness="0">
                    <Button.Template>
                        <ControlTemplate TargetType="Button">
                            <Border Name="border" Background="{TemplateBinding Background}" CornerRadius="0,0,0,12">
                                <ContentPresenter HorizontalAlignment="Center" VerticalAlignment="Center"/>
                            </Border>
                        </ControlTemplate>
                    </Button.Template>
                </Button>

                <Button Name="MainBtn" Grid.Row="1" Grid.Column="0" Grid.ColumnSpan="2" 
                        Content="분석 시작" Background="Transparent" BorderThickness="0" 
                        Foreground="White" FontSize="15" FontWeight="Bold" Cursor="Hand">
                    <Button.Template>
                        <ControlTemplate TargetType="Button">
                            <Border Name="MainBtnBorder" Background="{TemplateBinding Background}" CornerRadius="0,0,12,12">
                                <ContentPresenter HorizontalAlignment="Center" VerticalAlignment="Center"/>
                            </Border>
                        </ControlTemplate>
                    </Button.Template>
                </Button>
            </Grid>
        </Grid>
    </Border>
</Window>
"@

$reader = (New-Object System.Xml.XmlNodeReader $xaml)
$window = [Windows.Markup.XamlReader]::Load($reader)

# UI 요소 연결
$MainBtn = $window.FindName("MainBtn"); $SubBtn = $window.FindName("SubBtn"); $BtnClose = $window.FindName("BtnClose")
$StatusIcon = $window.FindName("StatusIcon")

# [추가됨] OS 정보 로드 및 표시
$txt_OSInfo = $window.FindName("txt_OSInfo")
try {
    $osName = (Get-CimInstance Win32_OperatingSystem).Caption.Replace("Microsoft ", "").Trim()
    $reg = Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion"
    $txt_OSInfo.Text = "$osName $($reg.DisplayVersion) ($($reg.CurrentBuild).$($reg.UBR))"
} catch { $txt_OSInfo.Text = "Windows System" }

$ModeText = $window.FindName("ModeText")
$TooltipContent = $ModeText.ToolTip.Content

$BigNumber = $window.FindName("BigNumber"); $LogText = $window.FindName("LogText")
$HintText = $window.FindName("HintText")
$ProgressBar = $window.FindName("ProgressBar"); $TitleBar = $window.FindName("TitleBar")
$ToggleDeepClean = $window.FindName("ToggleDeepClean"); $ToggleText = $window.FindName("ToggleText")

function Show-WarningDialog ($isDeepClean) {
    [xml]$warningXaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        Title="System Cleaner Warning" Height="240" Width="360" WindowStyle="None" ResizeMode="NoResize"
        AllowsTransparency="True" Background="Transparent" Topmost="True" WindowStartupLocation="CenterScreen">
    <Border CornerRadius="12" Background="#1E1E1E" BorderBrush="#333333" BorderThickness="1">
        <Grid>
            <Grid.RowDefinitions><RowDefinition Height="32"/><RowDefinition Height="*"/><RowDefinition Height="50"/></Grid.RowDefinitions>
            
            <Border Grid.Row="0" Background="#252525" CornerRadius="12,12,0,0" BorderBrush="#333" BorderThickness="0,0,0,1">
                <TextBlock Text="시스템 청소 경고" FontFamily="Malgun Gothic" Foreground="#AAA" VerticalAlignment="Center" Margin="15,0,0,0" FontWeight="SemiBold" FontSize="11"/>
            </Border>

            <StackPanel Grid.Row="1" VerticalAlignment="Center" HorizontalAlignment="Center" Margin="10">
                <StackPanel Orientation="Horizontal" HorizontalAlignment="Center" Margin="0,0,0,10">
                    <TextBlock Text="⚠️" FontSize="18" Foreground="#FFCC00" VerticalAlignment="Center" Margin="0,0,8,0"/>
                    <TextBlock Text="브라우저 강제 종료 알림" FontFamily="Malgun Gothic" FontSize="14" FontWeight="Bold" Foreground="White" VerticalAlignment="Center"/>
                </StackPanel>
                
                <TextBlock Text="모든 브라우저(Chrome, Edge 등)가 종료됩니다." FontFamily="Malgun Gothic" Foreground="#DDD" HorizontalAlignment="Center" FontSize="11" Margin="0,0,0,3"/>
                <TextBlock Text="작업 중인 내용을 저장하셨나요?" FontFamily="Malgun Gothic" Foreground="#DDD" HorizontalAlignment="Center" FontSize="11" Margin="0,0,0,15"/>
                
                <StackPanel Name="DeepWarning">
                    <TextBlock Text="※ WinSxS 및 정밀 정리는 시간이 소요됩니다." FontFamily="Malgun Gothic" Foreground="#FF6B6B" HorizontalAlignment="Center" FontSize="11" Margin="0,0,0,2"/>
                    <TextBlock Text="완료될 때까지 전원을 끄지 마세요." FontFamily="Malgun Gothic" Foreground="#FF6B6B" HorizontalAlignment="Center" FontSize="11"/>
                </StackPanel>
            </StackPanel>

            <Grid Grid.Row="2" Margin="0,0,0,10">
                <Grid.ColumnDefinitions><ColumnDefinition Width="*"/><ColumnDefinition Width="*"/></Grid.ColumnDefinitions>
                
                <Button Name="BtnNo" Grid.Column="0" Content="취소 (No)" Margin="20,0,10,0" Height="28"
                        Background="#333333" Foreground="#AAA" BorderThickness="0" Cursor="Hand" FontWeight="Bold" FontFamily="Malgun Gothic" FontSize="11">
                    <Button.Template>
                        <ControlTemplate TargetType="Button">
                            <Border Name="bdr" Background="{TemplateBinding Background}" CornerRadius="6">
                                <ContentPresenter HorizontalAlignment="Center" VerticalAlignment="Center"/>
                            </Border>
                            <ControlTemplate.Triggers>
                                <Trigger Property="IsMouseOver" Value="True">
                                    <Setter TargetName="bdr" Property="Background" Value="#444"/>
                                </Trigger>
                            </ControlTemplate.Triggers>
                        </ControlTemplate>
                    </Button.Template>
                </Button>

                <Button Name="BtnYes" Grid.Column="1" Content="진행 (Yes)" Margin="10,0,20,0" Height="28"
                        Background="#D9534F" Foreground="White" BorderThickness="0" Cursor="Hand" FontWeight="Bold" FontFamily="Malgun Gothic" FontSize="11">
                    <Button.Template>
                        <ControlTemplate TargetType="Button">
                            <Border Name="bdr" Background="{TemplateBinding Background}" CornerRadius="6">
                                <ContentPresenter HorizontalAlignment="Center" VerticalAlignment="Center"/>
                            </Border>
                            <ControlTemplate.Triggers>
                                <Trigger Property="IsMouseOver" Value="True">
                                    <Setter TargetName="bdr" Property="Background" Value="#C9302C"/>
                                </Trigger>
                            </ControlTemplate.Triggers>
                        </ControlTemplate>
                    </Button.Template>
                </Button>
            </Grid>
        </Grid>
    </Border>
</Window>
"@
    $reader = (New-Object System.Xml.XmlNodeReader $warningXaml)
    $warnWin = [Windows.Markup.XamlReader]::Load($reader)
    
    # [수정됨] 정밀 모드가 아니면 빨간 글씨(DeepWarning) 숨김
    $DeepWarning = $warnWin.FindName("DeepWarning")
    if (-not $isDeepClean) {
        $DeepWarning.Visibility = "Collapsed"
    }

    $boxState = @{ Result = $false }

    $warnWin.FindName("BtnYes").Add_Click({ 
        $boxState.Result = $true
        $warnWin.Close() 
    })
    
    $warnWin.FindName("BtnNo").Add_Click({ 
        $boxState.Result = $false
        $warnWin.Close() 
    })

    $warnWin.ShowDialog() | Out-Null
    return $boxState.Result
}

# =============================================================================
# 3. 로직 데이터 (Type='Heavy' 태그 추가됨, 나머지는 원본 동일)
# =============================================================================
# [수정됨] 툴팁 목록 만드는 함수 (글자 수 제한 제거)
function Get-CleanListText ($isDeep) {
    $list = @()
    foreach ($t in $CleanupTasks) {
        # 정밀 모드가 아닐 때 Heavy 태그 항목 건너뛰기
        if ($t.Type -eq "Heavy" -and -not $isDeep) { continue }
        
        # [삭제됨] if ($name.Length -gt 12) { ... }  <-- 이 부분이 글자를 자르던 범인입니다.
        
        # 이름 그대로 출력
        $list += "- $($t.Name)"
    }
    return ($list -join "`n")
}

function DoEvents {
    [System.Windows.Threading.Dispatcher]::CurrentDispatcher.Invoke([Action]{}, [System.Windows.Threading.DispatcherPriority]::Render)
}

$syncHash = [hashtable]::Synchronized(@{
    State = 0
    LogMessage = ""
    Progress = 0
    TotalSize = 0
    JobDone = $false
    DeepClean = $false  # 토글 상태 전달용 변수
})

$CleanupTasks = @(
    # [1] Windows 기본 (Heavy 태그 추가: WinSxS, 업데이트, Windows.old)
    @{ Name="Windows 업데이트 캐시"; Type="Heavy"; Path="$env:SystemRoot\SoftwareDistribution\Download"; Action={ 
        $services = "wuauserv", "bits", "cryptsvc"
        try {
            # 1. 서비스 안전하게 종료 (최대 10초 대기)
            foreach ($svcName in $services) {
                $svc = Get-Service $svcName -ErrorAction SilentlyContinue
                if ($svc -and $svc.Status -ne 'Stopped') {
                    Stop-Service $svcName -Force -ErrorAction SilentlyContinue
                    # 서비스가 완전히 멈출 때까지 잠시 대기 (Loop)
                    $limit = 0
                    while ((Get-Service $svcName).Status -ne 'Stopped' -and $limit -lt 20) {
                        Start-Sleep -Milliseconds 500
                        $limit++
                    }
                }
            }
            
            # 2. 파일 삭제
            Start-Sleep -Milliseconds 500
            Remove-Item "$($env:SystemRoot)\SoftwareDistribution\Download\*" -Recurse -Force -ErrorAction SilentlyContinue
        }
        finally {
            # 3. 서비스 다시 시작 (작업이 실패해도 무조건 실행됨)
            foreach ($svcName in $services) {
                Start-Service $svcName -ErrorAction SilentlyContinue
            }
        }
    } },
    
    # [WinSxS] UI 프리징 방지 적용
    @{ Name="WinSxS 정리(오래 걸림)"; Type="Heavy"; Warning=$true; Path=$null; Action={ 
        $proc = Start-Process -FilePath "dism.exe" -ArgumentList "/online /Cleanup-Image /StartComponentCleanup /NoRestart" -WindowStyle Hidden -PassThru
        while (-not $proc.HasExited) {
            [System.Windows.Threading.Dispatcher]::CurrentDispatcher.Invoke([Action]{}, [System.Windows.Threading.DispatcherPriority]::Background)
            Start-Sleep -Milliseconds 100
        }
    } },

    @{ Name="Windows.old 폴더"; Type="Heavy"; Warning=$true; Path="$env:SystemDrive\Windows.old"; Action={ 
        if (Test-Path "$($env:SystemDrive)\Windows.old") {
            # 1. 소유권 가져오기 (관리자 권한 획득)
            $null = Start-Process -FilePath "cmd.exe" -ArgumentList "/c takeown /F `"$($env:SystemDrive)\Windows.old`" /A /R /D Y" -WindowStyle Hidden -PassThru -Wait
            
            # 2. 모든 권한 부여 (삭제 가능하도록 설정)
            $null = Start-Process -FilePath "cmd.exe" -ArgumentList "/c icacls `"$($env:SystemDrive)\Windows.old`" /grant *S-1-5-32-544:F /T /C /Q" -WindowStyle Hidden -PassThru -Wait
            
            # [수정됨] 3. CMD로 고속 삭제 (렉 없음, 검은 창 안 뜸)
            # rmdir 명령어를 몰래 실행하고, 다 지워질 때까지 UI가 멈추지 않게 기다립니다.
            $proc = Start-Process -FilePath "cmd.exe" -ArgumentList "/c rmdir /s /q `"$($env:SystemDrive)\Windows.old`"" -WindowStyle Hidden -PassThru
            while (-not $proc.HasExited) {
                # 화면이 굳지 않게 계속 갱신(DoEvents)
                [System.Windows.Threading.Dispatcher]::CurrentDispatcher.Invoke([Action]{}, [System.Windows.Threading.DispatcherPriority]::Background)
                Start-Sleep -Milliseconds 100
            }
        }
    } },
	
    # [수정됨] 이벤트 로그는 무거우므로 정밀 모드(Heavy)로 분리
    @{ Name="이벤트 로그 정리"; Type="Heavy"; Path=$null; Action={ 
        Get-WinEvent -ListLog * -Force -ErrorAction SilentlyContinue | ForEach-Object { wevtutil cl $_.LogName 2>$null } 
    } },
    
    # [2] 임시 파일 (일반 - Light)
    @{ Name="DirectX 셰이더 캐시"; Path=@("C:\ProgramData\Microsoft\Windows\D3DSCache", "$env:LocalAppData\D3DSCache"); Action={ 
        Remove-Item "C:\ProgramData\Microsoft\Windows\D3DSCache\*" -Recurse -Force -ErrorAction SilentlyContinue
        Remove-Item "$($env:LocalAppData)\D3DSCache\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="시스템 임시 파일"; Path="C:\Windows\Temp"; Action={ 
        Remove-Item "C:\Windows\Temp\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="사용자 임시 파일"; Path="$env:TEMP"; Action={ 
        Remove-Item "$($env:TEMP)\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="프리페치 파일"; Path="$env:SystemRoot\Prefetch"; Action={ 
        # UI 멈춤 방지를 위해 CMD 프로세스로 분리
        $proc = Start-Process -FilePath "cmd.exe" -ArgumentList "/c del /F /Q `"$env:SystemRoot\Prefetch\*.*`"" -WindowStyle Hidden -PassThru
        while (-not $proc.HasExited) {
            [System.Windows.Threading.Dispatcher]::CurrentDispatcher.Invoke([Action]{}, [System.Windows.Threading.DispatcherPriority]::Background)
            Start-Sleep -Milliseconds 50
        }
    } },
    @{ Name="오류 보고(WER)"; Path="C:\ProgramData\Microsoft\Windows\WER"; Action={ 
        Remove-Item "C:\ProgramData\Microsoft\Windows\WER\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="전달 최적화 캐시"; Path="$env:SystemRoot\ServiceProfiles\NetworkService\AppData\Local\Microsoft\Windows\DeliveryOptimization\Cache"; Action={ 
        Remove-Item "$($env:SystemRoot)\ServiceProfiles\NetworkService\AppData\Local\Microsoft\Windows\DeliveryOptimization\Cache\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="USOPrivate 업데이트"; Path="C:\ProgramData\USOPrivate\UpdateStore"; Action={ 
        Remove-Item "C:\ProgramData\USOPrivate\UpdateStore\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="메모리 덤프"; Path=@("$env:SystemRoot\Minidump", "$env:SystemRoot\MEMORY.DMP"); Action={ 
        Remove-Item "$($env:SystemRoot)\Minidump\*" -Recurse -Force -ErrorAction SilentlyContinue
        Remove-Item "$($env:SystemRoot)\MEMORY.DMP" -Force -ErrorAction SilentlyContinue 
    } },
    
    # [3] 디펜더 및 로그
    @{ Name="Defender 보호 기록"; Path="C:\ProgramData\Microsoft\Windows Defender\Scans\History\Service\DetectionHistory"; Action={ 
        if (Get-Module -ListAvailable -Name "Defender") { Import-Module Defender -Force -ErrorAction SilentlyContinue; Remove-MpThreat -All -ErrorAction SilentlyContinue }
        Remove-Item "C:\ProgramData\Microsoft\Windows Defender\Scans\History\Service\DetectionHistory\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    # [수정됨] DNS는 가벼워서 매일 수행
    @{ Name="DNS 캐시"; Path=$null; Action={ 
        ipconfig /flushdns | Out-Null
    } },

    # [4] 그래픽/글꼴
    @{ Name="글꼴 캐시"; Path="$env:LocalAppData\Microsoft\FontCache"; Action={ 
        Stop-Service FontCache -ErrorAction SilentlyContinue
        Remove-Item "$($env:LocalAppData)\Microsoft\FontCache\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },

    # [5] 브라우저 (프로세스 종료 로직 포함)
    @{ Name="Microsoft Store 캐시"; Path="$env:LocalAppData\Packages\Microsoft.WindowsStore_*\LocalCache"; IsWildcard=$true; Action={ 
        Get-ChildItem "$($env:LocalAppData)\Packages\Microsoft.WindowsStore_*\LocalCache" -Recurse -Force | Remove-Item -Force -Recurse -ErrorAction SilentlyContinue 
    } },
    @{ Name="Edge 브라우저 캐시"; Path=@(
            "$env:LocalAppData\Microsoft\Edge\User Data\Default\Cache",
            "$env:LocalAppData\Microsoft\Edge\User Data\Default\Code Cache",
            "$env:LocalAppData\Microsoft\Edge\User Data\Default\GPUCache",
            "$env:LocalAppData\Microsoft\Edge\User Data\Default\Media Cache"
        ); Action={ 
        Stop-Process -Name "msedge" -Force -ErrorAction SilentlyContinue
        Start-Sleep -Milliseconds 500
        @("Cache", "Code Cache", "GPUCache", "Media Cache") | ForEach-Object { Remove-Item "$($env:LocalAppData)\Microsoft\Edge\User Data\Default\$_*" -Recurse -Force -ErrorAction SilentlyContinue }
    } },
    @{ Name="Whale 브라우저 캐시"; Path=@(
            "$env:LocalAppData\Naver\Naver Whale\User Data\Default\Cache",
            "$env:LocalAppData\Naver\Naver Whale\User Data\Default\Code Cache",
            "$env:LocalAppData\Naver\Naver Whale\User Data\Default\GPUCache",
            "$env:LocalAppData\Naver\Naver Whale\User Data\Default\Media Cache"
        ); Action={ 
        Stop-Process -Name "whale" -Force -ErrorAction SilentlyContinue
        Start-Sleep -Milliseconds 500
        @("Cache", "Code Cache", "GPUCache", "Media Cache") | ForEach-Object { Remove-Item "$($env:LocalAppData)\Naver\Naver Whale\User Data\Default\$_*" -Recurse -Force -ErrorAction SilentlyContinue }
    } },
    @{ Name="Chrome 브라우저 캐시"; Path=@(
            "$env:LocalAppData\Google\Chrome\User Data\Default\Cache",
            "$env:LocalAppData\Google\Chrome\User Data\Default\Code Cache",
            "$env:LocalAppData\Google\Chrome\User Data\Default\GPUCache",
            "$env:LocalAppData\Google\Chrome\User Data\Default\Media Cache"
        ); Action={ 
        Stop-Process -Name "chrome" -Force -ErrorAction SilentlyContinue
        Start-Sleep -Milliseconds 500
        @("Cache", "Code Cache", "GPUCache", "Media Cache") | ForEach-Object { Remove-Item "$($env:LocalAppData)\Google\Chrome\User Data\Default\$_*" -Recurse -Force -ErrorAction SilentlyContinue }
    } },
	
    # [추가] RDP & Adobe
    @{ Name="원격 데스크톱(RDP) 캐시"; Path="$env:LocalAppData\Microsoft\Terminal Server Client\Cache"; Action={ 
        Remove-Item "$($env:LocalAppData)\Microsoft\Terminal Server Client\Cache\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="Adobe Acrobat 캐시"; Path="$env:LocalAppData\Adobe\Acrobat\*\Cache"; IsWildcard=$true; Action={ 
        Get-ChildItem "$($env:LocalAppData)\Adobe\Acrobat\*\Cache" -Recurse -Force -ErrorAction SilentlyContinue | Remove-Item -Force -Recurse -ErrorAction SilentlyContinue
    } },
	
    # [6] 탐색기/휴지통 (안전장치 강화)
    @{ Name="탐색기/썸네일 캐시"; Path="$env:LocalAppData\Microsoft\Windows\Explorer"; Special="Explorer"; Action={ 
        try {
            Stop-Process -Name "explorer" -Force -ErrorAction SilentlyContinue
            Start-Sleep -Milliseconds 500
            Get-ChildItem "$($env:LocalAppData)\Microsoft\Windows\Explorer\thumbcache_*.db" -Force -ErrorAction SilentlyContinue | Remove-Item -Force -ErrorAction SilentlyContinue
            Get-ChildItem "$($env:LocalAppData)\Microsoft\Windows\Explorer\IconCache*" -Force -ErrorAction SilentlyContinue | Remove-Item -Force -ErrorAction SilentlyContinue
        }
        finally {
            # 탐색기가 죽어있으면 무조건 살림
            if (-not (Get-Process -Name "explorer" -ErrorAction SilentlyContinue)) {
                Start-Process explorer.exe
            }
        }
    } },
    @{ Name="시스템 복원 및 MRU"; Path=$null; Special="ShadowCopy"; Action={ 
        vssadmin delete shadows /all /quiet
        Remove-ItemProperty -Path "HKCU:\Software\Microsoft\Windows\CurrentVersion\Explorer\RunMRU" -Name "*" -ErrorAction SilentlyContinue 
    } },
    @{ Name="휴지통 비우기"; Path=$null; Special="RecycleBin"; Action={ 
        Clear-RecycleBin -Force -ErrorAction SilentlyContinue 
    } },  # <--- [중요] 여기에 콤마(,)가 꼭 있어야 다음 게임 캐시로 이어집니다.

    # [추가됨] 게임 런처 캐시 (일반 모드)
    @{ Name="Steam 런처 캐시"; Path="$env:LocalAppData\Steam\htmlcache"; Action={ 
        Remove-Item "$($env:LocalAppData)\Steam\htmlcache\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="Battle.net 캐시"; Path="$env:ProgramData\Battle.net\Agent\data"; Action={ 
        Remove-Item "$($env:ProgramData)\Battle.net\Agent\data\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } },
    @{ Name="Epic Games 캐시"; Path="$env:LocalAppData\EpicGamesLauncher\Saved\webcache"; Action={ 
        Remove-Item "$($env:LocalAppData)\EpicGamesLauncher\Saved\webcache\*" -Recurse -Force -ErrorAction SilentlyContinue 
    } }
)

# =============================================================================
# 4. 타이머 및 이벤트 처리 (토글 이벤트 추가됨)
# =============================================================================
$TitleBar.Add_MouseLeftButtonDown({ $window.DragMove() })
$BtnClose.Add_Click({ $window.Close() })

# [NEW] 토글 클릭 이벤트: 색상 변경 및 상태 텍스트 변경
$ToggleDeepClean.Add_Click({
    if ($ToggleDeepClean.IsChecked) {
        # ON 상태: 주황색 경고 모드
        $StatusIcon.Foreground = "#FF9500" 
        $ToggleText.Foreground = "#FF9500"
        $ToggleText.Text = "정밀 정리 (시간 소요됨)"
        $HintText.Foreground = "#FF9500"
        $MainBtn.Background = "#FF9500" # 버튼 색상도 약간의 힌트를 줌
	$ModeText.Text = "정밀 정리 모드"
    } else {
        # OFF 상태: 기본 파란색 모드
        $StatusIcon.Foreground = "#0078D7"
        $ToggleText.Foreground = "#666"
        $ToggleText.Text = "정밀 정리 (오래 걸림)"
        $HintText.Foreground = "#4CC2FF"
        $MainBtn.Background = "Transparent"
	$ModeText.Text = "일반 정리 모드"
    }
    $TooltipContent.Text = Get-CleanListText ($ToggleDeepClean.IsChecked)
})

$timer = New-Object System.Windows.Threading.DispatcherTimer
$timer.Interval = [TimeSpan]::FromMilliseconds(50)
$timer.Add_Tick({
    # State 1: 분석 중
    if ($syncHash.State -eq 1) {
        $ProgressBar.Value = $syncHash.Progress
        $LogText.Text = $syncHash.LogMessage
        
        if ($syncHash.JobDone) {
            $syncHash.State = 2
            $syncHash.JobDone = $false
            $syncHash.LogMessage = "분석이 완료되었습니다." 
            
            $BtnClose.Visibility = "Visible"
            $MainBtn.IsEnabled = $true
            $MainBtn.Content = "정리 시작"
            
            # 버튼 색상 및 텍스트 처리
            if ($ToggleDeepClean.IsChecked) {
                # [Case 1] 정밀 모드 (주황)
                $MainBtn.Background = "#FF9500"
                $MainBtn.Foreground = "White"
                
                # 정밀 모드일 때만 힌트(경고) 텍스트 표시
                $HintText.Text = "※ 정밀 정리 모드입니다.`n시간이 다소 소요될 수 있습니다."
                $HintText.Visibility = "Visible"
            } else {
                # [Case 2] 일반 모드 (파랑)
                $MainBtn.Background = "#4CC2FF"  
                $MainBtn.Foreground = "#111111"  
                
                # [수정됨] 일반 모드에서는 힌트 텍스트를 아예 숨김 (깔끔하게)
                $HintText.Visibility = "Collapsed"
            }
            
            $ModeText.Text = "확보 가능 용량"
            $StatusIcon.Visibility = "Collapsed"
            $BigNumber.Visibility = "Visible"
            $mb = $syncHash.TotalSize / 1MB
            $BigNumber.Text = if ($mb -gt 1024) { "{0:N2} GB" -f ($mb/1024) } else { "{0:N2} MB" -f $mb }
            
            $LogText.Text = ""
	    $LogText.Visibility = "Collapsed"
            
            # 분석 끝나면 토글 못 바꾸게 잠금
            $ToggleDeepClean.IsEnabled = $false
        }
    }
})

$SubBtn.Add_Click({
    # 초기화
    $syncHash.State = 0
    $syncHash.Progress = 0
    $syncHash.TotalSize = 0
    $syncHash.JobDone = $false
    
    $BtnClose.Visibility = "Visible"
    $SubBtn.Visibility = "Collapsed"
    
    $MainBtn.SetValue([System.Windows.Controls.Grid]::ColumnProperty, 0)
    $MainBtn.SetValue([System.Windows.Controls.Grid]::ColumnSpanProperty, 2)
    
    $border = $MainBtn.Template.FindName("MainBtnBorder", $MainBtn)
    if ($border) { $border.CornerRadius = New-Object System.Windows.CornerRadius(0,0,12,12) }
    
    $MainBtn.Content = "분석 시작"
    $MainBtn.Background = "Transparent"
    $MainBtn.Foreground = "White"
    $MainBtn.IsEnabled = $true 
    
    $StatusIcon.Visibility = "Visible"
    $StatusIcon.Text = "🛡️"
    
    # 토글 상태에 따라 아이콘 색상 복구
    if ($ToggleDeepClean.IsChecked) {
        $StatusIcon.Foreground = "#FF9500"
        $ModeText.Text = "정밀 정리 모드"
    } else {
        $StatusIcon.Foreground = "#0078D7"
        $ModeText.Text = "일반 정리 모드"
    }

    $BigNumber.Visibility = "Collapsed"
    $HintText.Visibility = "Collapsed"
    $ProgressBar.Value = 0
    $ProgressBar.Visibility = "Visible"
    
    # 재검색 시 토글 다시 활성화
    $ToggleDeepClean.IsEnabled = $true

    $MainBtn.RaiseEvent((New-Object System.Windows.RoutedEventArgs ([System.Windows.Controls.Button]::ClickEvent)))
})

$MainBtn.Add_Click({
    if ($syncHash.State -eq 0) {
        # --- [State 1] 분석 시작 ---
	$txt_OSInfo.Visibility = 'Collapsed'
        $syncHash.State = 1
        $BtnClose.Visibility = "Collapsed"
        
        $syncHash.DeepClean = $ToggleDeepClean.IsChecked
        
        if ($syncHash.DeepClean) {
            $ProgressBar.Foreground = "#FF9500" # 주황
        } else {
            $ProgressBar.Foreground = "#4CC2FF" # 하늘
        }

        DoEvents
        $MainBtn.IsEnabled = $false
        $MainBtn.Content = "분석 중..."
        
        # [수정 완료] SubText -> ModeText
        $ModeText.Text = "정리 대상 스캔 중"
        
        $ToggleDeepClean.IsEnabled = $false
        
        $ps = [powershell]::Create().AddScript({
            $total = 0; $cnt = 0
            foreach ($t in $CleanupTasks) {
                if ($t.Type -eq "Heavy" -and -not $syncHash.DeepClean) { 
                    $cnt++
                    $syncHash.Progress = ($cnt / $CleanupTasks.Count) * 100
                    continue 
                }

                $syncHash.LogMessage = "분석 중: $($t.Name)"
                if ($t.Path) {
                    $paths = if ($t.Path -is [array]) { $t.Path } else { @($t.Path) }
                    foreach ($p in $paths) {
                        try {
                            if ($t.IsWildcard) {
                                $expandedPaths = Get-ChildItem $p -Recurse -Force -ErrorAction SilentlyContinue | Select-Object -ExpandProperty FullName
                                if ($expandedPaths) { $total += (Get-ChildItem -LiteralPath $expandedPaths -Recurse -Force -ErrorAction SilentlyContinue | Measure-Object -Property Length -Sum).Sum }
                            }
                            elseif ($t.Special -eq "Explorer") {
                                if (Test-Path $p) {
                                    $items = Get-ChildItem $p -Include "thumbcache_*.db","IconCache*" -Recurse -Force -ErrorAction SilentlyContinue
                                    $total += ($items | Measure-Object -Property Length -Sum).Sum
                                }
                            }
                            elseif (Test-Path $p) {
                                $items = Get-ChildItem $p -Recurse -Force -ErrorAction SilentlyContinue | Where-Object { -not ($_.Attributes -band [System.IO.FileAttributes]::ReparsePoint) }
                                $total += ($items | Measure-Object -Property Length -Sum).Sum
                            }
                        } catch {}
                    }
                }
                if ($t.Special -eq "ShadowCopy") {
                    try { $shadow = Get-CimInstance -ClassName Win32_ShadowStorage -ErrorAction SilentlyContinue; if ($shadow) { $total += ($shadow | Measure-Object -Property UsedSpace -Sum).Sum } } catch {}
                }
                if ($t.Special -eq "RecycleBin") {
                    try { $recycle = Get-ChildItem "C:\`$Recycle.Bin" -Recurse -Force -ErrorAction SilentlyContinue; $total += ($recycle | Measure-Object -Property Length -Sum).Sum } catch {}
                }
                $cnt++
                $syncHash.Progress = ($cnt / $CleanupTasks.Count) * 100
                Start-Sleep -Milliseconds 10
            }
            $syncHash.TotalSize = $total
            $syncHash.JobDone = $true
        })
        $ps.Runspace.SessionStateProxy.SetVariable("syncHash", $syncHash)
        $ps.Runspace.SessionStateProxy.SetVariable("CleanupTasks", $CleanupTasks)
        $null = $ps.BeginInvoke()
    }
    elseif ($syncHash.State -eq 2) {
        $userChoice = Show-WarningDialog -isDeepClean $syncHash.DeepClean
        if ($userChoice -eq $false) { return }

        # --- [State 3] 청소 시작 ---
        $syncHash.State = 3
        $BtnClose.Visibility = "Collapsed"
        DoEvents
	
	$LogText.Visibility = "Visible"
        
        $BigNumber.Visibility = "Collapsed"
        $HintText.Visibility = "Collapsed"
        
        $MainBtn.IsEnabled = $false
        $MainBtn.Content = "청소 중..."
        
        # [수정 완료] SubText -> ModeText
        $ModeText.Text = "시스템 최적화 진행 중"
        
        $MainBtn.Background = "#333333" 
        $MainBtn.Foreground = "#888888"
        
        $ProgressBar.Visibility = "Visible"
        $ProgressBar.Value = 0
        DoEvents

        $cnt = 0
        foreach ($t in $CleanupTasks) {
            if ($t.Type -eq "Heavy" -and -not $syncHash.DeepClean) { 
                $cnt++
                $ProgressBar.Value = ($cnt / $CleanupTasks.Count) * 100
                continue 
            }

            if ($t.Name -like "*WinSxS*") {
                $LogText.Foreground = [System.Windows.Media.BrushConverter]::new().ConvertFrom("#FF6B6B")
                $LogText.Text = "업데이트 구성 요소 정리 중... (오래 걸립니다!)"
            } elseif ($t.Name -like "*브라우저*") {
                $LogText.Foreground = [System.Windows.Media.Brushes]::White
                $LogText.Text = "브라우저 및 캐시 정리 중..."
            } else {
                $LogText.Foreground = [System.Windows.Media.BrushConverter]::new().ConvertFrom("#AAAAAA")
                $LogText.Text = "삭제 중: $($t.Name)"
            }
            DoEvents 
            try { & $t.Action } catch {}
            $cnt++
            $ProgressBar.Value = ($cnt / $CleanupTasks.Count) * 100
            DoEvents 
            Start-Sleep -Milliseconds 50 
        }
        
        # --- 완료 후 처리 ---
        $syncHash.State = 4
        $LogText.Text = "정리를 완료하였습니다."
        $LogText.Foreground = [System.Windows.Media.Brushes]::Gray
        
        $BtnClose.Visibility = "Visible"
        $SubBtn.Visibility = "Visible"
        
        $MainBtn.SetValue([System.Windows.Controls.Grid]::ColumnProperty, 1)
        $MainBtn.SetValue([System.Windows.Controls.Grid]::ColumnSpanProperty, 1)
        
        $border = $MainBtn.Template.FindName("MainBtnBorder", $MainBtn)
        if ($border) { $border.CornerRadius = New-Object System.Windows.CornerRadius(0,0,12,0) }
        
        $MainBtn.IsEnabled = $true
        $MainBtn.Content = "지금 재부팅"
        
        $ProgressBar.Visibility = "Collapsed" 
        
        # [중요 수정] 여기서 SubText가 남아있어서 오류가 났던 겁니다!
        $ModeText.Text = "최적화 완료!"
        
        $StatusIcon.Text = "✅"
        $StatusIcon.Visibility = "Visible"
        $BigNumber.Visibility = "Collapsed"
        $ProgressBar.Value = 100

        if ($syncHash.DeepClean) {
            $MainBtn.Background = "#FF9500" 
            $StatusIcon.Foreground = "#FF9500" 
        } else {
            $MainBtn.Background = "#28A745" 
            $StatusIcon.Foreground = "#28A745" 
        }
        $MainBtn.Foreground = "White"
    }
    elseif ($syncHash.State -eq 4) {
        Restart-Computer -Force
    }
})
$TooltipContent.Text = Get-CleanListText $false

$timer.Start()
$window.ShowDialog() | Out-Null